<?php

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class AdminDivisionsUpdater
{
    public function run()
    {
        $this->commandInfo("Starting Administrative Divisions Update (2025)");
        
        // Disable foreign key checks temporarily
        DB::statement('SET FOREIGN_KEY_CHECKS=0');
        
        try {
            // Clear existing data
            $this->clearExistingData();
            
            // Add new data
            $this->addProvinces();
            $this->addDistricts();
            $this->addCommunes();
            $this->addVillages();
            
            $this->commandSuccess("✅ Administrative divisions have been successfully updated to 2025 data!");
            
        } catch (\Exception $e) {
            $this->commandError("❌ Error updating administrative divisions: " . $e->getMessage());
            throw $e;
        } finally {
            // Re-enable foreign key checks
            DB::statement('SET FOREIGN_KEY_CHECKS=1');
        }
    }
    
    protected function clearExistingData()
    {
        $this->commandInfo("Clearing existing data...");
        
        // Clear tables in the correct order to respect foreign key constraints
        DB::table('village')->truncate();
        DB::table('commune')->truncate();
        DB::table('district')->truncate();
        DB::table('province')->truncate();
        
        $this->commandSuccess("✅ Cleared existing administrative divisions data");
    }
    
    protected function addProvinces()
    {
        $this->commandInfo("Adding provinces...");
        
        $provinces = [
            ['id' => 1, 'name' => 'ភ្នំពេញ', 'created_at' => now(), 'updated_at' => now()],
            ['id' => 2, 'name' => 'កណ្ដាល', 'created_at' => now(), 'updated_at' => now()],
            ['id' => 3, 'name' => 'កំពង់ចាម', 'created_at' => now(), 'updated_at' => now()],
            // Add all 25 provinces here
            // ...
        ];
        
        DB::table('province')->insert($provinces);
        $this->commandSuccess("✅ Added " . count($provinces) . " provinces");
    }
    
    protected function addDistricts()
    {
        $this->commandInfo("Adding districts...");
        
        $districts = [
            // Phnom Penh
            ['id' => 101, 'province_id' => 1, 'name' => 'ជ្រោយចង្វារ', 'created_at' => now(), 'updated_at' => now()],
            ['id' => 102, 'province_id' => 1, 'name' => 'ដូនពេញ', 'created_at' => now(), 'updated_at' => now()],
            // Add all districts for each province
            // ...
            
            // Kampong Thom
            ['id' => 601, 'province_id' => 5, 'name' => 'បារាយណ៍', 'created_at' => now(), 'updated_at' => now()],
            ['id' => 602, 'province_id' => 5, 'name' => 'កំពង់ស្វាយ', 'created_at' => now(), 'updated_at' => now()],
            ['id' => 603, 'province_id' => 5, 'name' => 'ក្រុងស្ទឹងសែន', 'created_at' => now(), 'updated_at' => now()],
            ['id' => 604, 'province_id' => 5, 'name' => 'ប្រាសាទបល្ល័ង្គ', 'created_at' => now(), 'updated_at' => now()],
            ['id' => 605, 'province_id' => 5, 'name' => 'ប្រាសាទសំបូរ', 'created_at' => now(), 'updated_at' => now()],
            ['id' => 606, 'province_id' => 5, 'name' => 'សណ្ដាន់', 'created_at' => now(), 'updated_at' => now()],
            ['id' => 607, 'province_id' => 5, 'name' => 'សន្ទុក', 'created_at' => now(), 'updated_at' => now()],
            ['id' => 608, 'province_id' => 5, 'name' => 'ស្ទោង', 'created_at' => now(), 'updated_at' => now()],
            ['id' => 609, 'province_id' => 5, 'name' => 'តាំងគោក', 'created_at' => now(), 'updated_at' => now()],
        ];
        
        DB::table('district')->insert($districts);
        $this->commandSuccess("✅ Added " . count($districts) . " districts");
    }
    
    protected function addCommunes()
    {
        $this->commandInfo("Adding communes...");
        
        $communes = [
            // Add communes for each district
            // Format: [id, district_id, name, created_at, updated_at]
            
            // Example for Tang Kouk district (ID: 609)
            [60901, 609, 'តាំងគោក', now(), now()],
            [60902, 609, 'ព្រៃធំ', now(), now()],
            // Add more communes as needed
        ];
        
        // Insert in chunks to avoid memory issues
        foreach (array_chunk($communes, 100) as $chunk) {
            DB::table('commune')->insert($chunk);
        }
        
        $this->commandSuccess("✅ Added " . count($communes) . " communes");
    }
    
    protected function addVillages()
    {
        $this->commandInfo("Adding villages...");
        
        $villages = [
            // Add villages for each commune
            // Format: [commune_id, name, created_at, updated_at]
            
            // Example for Tang Kouk commune (ID: 60901)
            [60901, 'ភូមិ ១', now(), now()],
            [60901, 'ភូមិ ២', now(), now()],
            // Add more villages as needed
        ];
        
        // Insert in chunks to avoid memory issues
        foreach (array_chunk($villages, 100) as $chunk) {
            DB::table('village')->insert($chunk);
        }
        
        $this->commandSuccess("✅ Added " . count($villages) . " villages");
    }
    
    // Helper methods for console output
    protected function commandInfo($message)
    {
        echo "\n\033[36m$message\033[0m\n";
    }
    
    protected function commandSuccess($message)
    {
        echo "\033[32m$message\033[0m\n";
    }
    
    protected function commandError($message)
    {
        echo "\033[31m$message\033[0m\n";
    }
}

// Run the updater
$updater = new AdminDivisionsUpdater();
$updater->run();
