class FormValidator {
    constructor(formSelector) {
        this.form = document.querySelector(formSelector);
        this.fields = [];
        this.init();
    }

    init() {
        if (!this.form) return;

        // Collect all required fields
        this.collectRequiredFields();
        
        // Add event listeners
        this.form.addEventListener('submit', (e) => this.handleSubmit(e));
        
        // Add real-time validation
        this.fields.forEach(field => {
            if (field.element) {
                field.element.addEventListener('blur', () => this.validateField(field));
                field.element.addEventListener('focus', () => this.clearError(field));
                field.element.addEventListener('input', () => this.clearError(field));
                field.element.addEventListener('change', () => this.clearError(field));
                
                // Special handling for Khmer date picker
                if (field.type === 'date' && field.element.name) {
                    const daySelect = document.querySelector(`select[name="${field.element.name}_day"]`);
                    const monthSelect = document.querySelector(`select[name="${field.element.name}_month"]`);
                    const yearSelect = document.querySelector(`select[name="${field.element.name}_year"]`);
                    
                    [daySelect, monthSelect, yearSelect].forEach(select => {
                        if (select) {
                            select.addEventListener('focus', () => this.clearError(field));
                            select.addEventListener('change', () => this.clearError(field));
                        }
                    });
                }
                
                // Special handling for radio buttons
                if (field.type === 'radio' && field.group) {
                    field.group.forEach(radio => {
                        radio.addEventListener('focus', () => this.clearError(field));
                        radio.addEventListener('change', () => this.clearError(field));
                    });
                }
            }
        });
    }

    collectRequiredFields() {
        // Text inputs
        const textInputs = this.form.querySelectorAll('input[type="text"], input[type="tel"], input[type="password"], input[type="email"]');
        textInputs.forEach(input => {
            if (this.isRequiredByLabel(input)) {
                this.fields.push({
                    element: input,
                    type: 'text',
                    name: input.name || input.id,
                    label: this.getFieldLabel(input)
                });
            }
        });

        // Select elements
        const selects = this.form.querySelectorAll('select');
        selects.forEach(select => {
            if (this.isRequiredByLabel(select)) {
                this.fields.push({
                    element: select,
                    type: 'select',
                    name: select.name || select.id,
                    label: this.getFieldLabel(select)
                });
            }
        });

        // Radio groups
        const radioGroups = this.groupRadioButtons();
        radioGroups.forEach(group => {
            if (this.isRequiredByLabel(group[0])) {
                this.fields.push({
                    element: group[0],
                    type: 'radio',
                    name: group[0].name,
                    label: this.getFieldLabel(group[0]),
                    group: group
                });
            }
        });

        // File inputs
        const fileInputs = this.form.querySelectorAll('input[type="file"]');
        fileInputs.forEach(input => {
            if (this.isRequiredByLabel(input)) {
                this.fields.push({
                    element: input,
                    type: 'file',
                    name: input.name || input.id,
                    label: this.getFieldLabel(input)
                });
            }
        });

        // Date picker inputs - handle both old and new
        const dateInputs = this.form.querySelectorAll('input[readonly]');
        dateInputs.forEach(input => {
            if (this.isRequiredByLabel(input)) {
                this.fields.push({
                    element: input,
                    type: 'date',
                    name: input.name || input.id,
                    label: this.getFieldLabel(input)
                });
            }
        });

        // New Khmer date picker - detect by hidden inputs
        const khmerDateInputs = this.form.querySelectorAll('input.khmer-date-hidden');
        khmerDateInputs.forEach(input => {
            if (this.isRequiredByLabel(input)) {
                this.fields.push({
                    element: input,
                    type: 'date',
                    name: input.name || input.id,
                    label: this.getFieldLabel(input)
                });
            }
        });
    }

    isRequiredByLabel(element) {
        const fieldContainer = element.closest('label') || element.closest('.field');
        if (fieldContainer) {
            const labelSpan = fieldContainer.querySelector('span');
            if (labelSpan) {
                const labelText = labelSpan.textContent;
                // Only require fields that explicitly have asterisk (*)
                return labelText.includes('*');
            }
        }
        return false;
    }

    getFieldLabel(element) {
        const fieldContainer = element.closest('label') || element.closest('.field');
        if (fieldContainer) {
            const labelSpan = fieldContainer.querySelector('span');
            return labelSpan ? labelSpan.textContent.trim() : '';
        }
        return element.name || element.id || '';
    }

    groupRadioButtons() {
        const radioGroups = {};
        const radios = this.form.querySelectorAll('input[type="radio"]');
        
        radios.forEach(radio => {
            if (!radioGroups[radio.name]) {
                radioGroups[radio.name] = [];
            }
            radioGroups[radio.name].push(radio);
        });
        
        return Object.values(radioGroups);
    }

    validateField(field) {
        let isValid = true;
        let errorMessage = '';

        switch (field.type) {
            case 'text':
            case 'tel':
                isValid = field.element.value.trim() !== '';
                // Special validation for phone number
                if (field.element.name === 'phone_number' && field.element.value.trim() !== '') {
                    const phoneRegex = /^[0-9]{8,15}$/;
                    isValid = phoneRegex.test(field.element.value.trim());
                    if (!isValid) {
                        errorMessage = 'លេខទូរស័ព្ទត្រូវតែជាលេខគត់ និងមានចាន់ពី 8 ដល់ 15 ខ្ទង់!';
                    } else {
                        errorMessage = `${field.label} ត្រូវបំបេញ`;
                    }
                } else {
                    errorMessage = `${field.label} ត្រូវបំបេញ`;
                }
                break;
            
            case 'select':
                isValid = field.element.value !== '';
                errorMessage = `សូមជ្រើសរើស ${field.label}`;
                break;
            
            case 'radio':
                if (field.group) {
                    isValid = field.group.some(radio => radio.checked);
                    // Special message for monk/nun type
                    if (field.label.includes('ភិក្ខុ') || field.label.includes('សាមណេរ')) {
                        errorMessage = 'សូមជ្រើសរើសភេទភិក្ខុ/សាមណេរ';
                    } else {
                        errorMessage = `សូមជ្រើសរើស ${field.label}`;
                    }
                }
                break;
            
            case 'file':
                isValid = field.element.files.length > 0;
                // Special message for image upload
                if (field.label.includes('រូប') || field.label.includes('រូបភាព')) {
                    errorMessage = 'សូមបញ្ចូលរូបភាព';
                } else {
                    errorMessage = `សូមជ្រើសរើស ${field.label}`;
                }
                break;
            
            case 'date':
                // Handle both old date picker and new Khmer date picker
                if (field.picker) {
                    // Old date picker with single input
                    const selects = field.picker.querySelectorAll('select');
                    isValid = Array.from(selects).every(select => select.value !== '');
                    errorMessage = `សូមជ្រើសរើស ${field.label}`;
                } else if (field.element && field.element.name) {
                    // New Khmer date picker with separate selects
                    const daySelect = document.querySelector(`select[name="${field.element.name}_day"]`);
                    const monthSelect = document.querySelector(`select[name="${field.element.name}_month"]`);
                    const yearSelect = document.querySelector(`select[name="${field.element.name}_year"]`);
                    
                    if (daySelect && monthSelect && yearSelect) {
                        isValid = daySelect.value !== '' && monthSelect.value !== '' && yearSelect.value !== '';
                        errorMessage = `សូមជ្រើសរើស ${field.label}`;
                    } else {
                        // Fallback to checking hidden input
                        isValid = field.element.value !== '';
                        errorMessage = `សូមជ្រើសរើស ${field.label}`;
                    }
                }
                break;
        }

        if (!isValid) {
            this.showError(field, errorMessage);
        } else {
            this.clearError(field);
        }

        return isValid;
    }

    showError(field, message) {
        this.clearError(field);

        // Add error class to field container
        const fieldContainer = field.element.closest('.field') || field.element.closest('.form-group');
        if (fieldContainer) {
            fieldContainer.classList.add('field-error');
        }

        // Create error message element
        const errorElement = document.createElement('div');
        errorElement.className = 'error-message';
        errorElement.textContent = message;
        errorElement.style.cssText = `
            color: #dc3545;
            font-size: 0.875rem;
            margin-top: 4px;
            display: block;
            animation: slideDown 0.3s ease-out;
        `;

        // Insert error message after the field
        if (fieldContainer) {
            fieldContainer.appendChild(errorElement);
        }

        // Add shake animation to input
        if (field.element) {
            field.element.style.animation = 'shake 0.5s';
            setTimeout(() => {
                field.element.style.animation = '';
            }, 500);
        }
    }

    clearError(field) {
        // Remove error class from field container
        const fieldContainer = field.element.closest('.field') || field.element.closest('.form-group');
        if (fieldContainer) {
            fieldContainer.classList.remove('field-error');
        }

        // Remove error message
        const errorElement = fieldContainer?.querySelector('.error-message');
        if (errorElement) {
            errorElement.remove();
        }
    }

    validateAll() {
        let isValid = true;
        this.fields.forEach(field => {
            if (!this.validateField(field)) {
                isValid = false;
            }
        });
        return isValid;
    }

    handleSubmit(e) {
        const isValid = this.validateAll();
        if (!isValid) {
            e.preventDefault();
            
            // Focus on first error field
            const firstErrorField = this.form.querySelector('.field-error input, .field-error select, .field-error input[readonly]');
            if (firstErrorField) {
                firstErrorField.focus();
            }
        }
    }
}

// Add CSS animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideDown {
        from {
            opacity: 0;
            transform: translateY(-10px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    @keyframes slideUp {
        from {
            opacity: 1;
            transform: translateY(0);
        }
        to {
            opacity: 0;
            transform: translateY(-10px);
        }
    }

    @keyframes shake {
        0%, 100% { transform: translateX(0); }
        10%, 30%, 50%, 70%, 90% { transform: translateX(-5px); }
        20%, 40%, 60%, 80% { transform: translateX(5px); }
    }

    .field-error input,
    .field-error select,
    .field-error input[readonly] {
        border-color: #dc3545 !important;
        box-shadow: 0 0 0 2px rgba(220, 53, 69, 0.2) !important;
    }

    .field-error .date-select {
        border-color: #dc3545 !important;
    }

    .field-error .radio-pill span {
        border-color: #dc3545 !important;
        box-shadow: 0 0 0 2px rgba(220, 53, 69, 0.2) !important;
    }

    .field-error input:focus,
    .field-error select:focus,
    .field-error input[readonly]:focus {
        border-color: #dc3545 !important;
        box-shadow: 0 0 0 3px rgba(220, 53, 69, 0.3) !important;
    }
`;
document.head.appendChild(style);

// Initialize validators when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    // Initialize for the main registration form
    const registrationForm = document.querySelector('form');
    if (registrationForm) {
        new FormValidator('form');
    }

    // Remove the custom file validation since FormValidator handles it
    const existingFileValidation = document.querySelector('script[data-file-validation]');
    if (existingFileValidation) {
        existingFileValidation.remove();
    }
});
