<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

class VillageApiSeeder extends Seeder
{
    protected $baseUrl = 'https://data.mef.gov.kh/api/v1/public-datasets/pd_66a8603a00604c000123e147/json';
    protected $pageSize = 200;

    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $this->command->info('Starting Village API Seeder...');
        $this->command->info('Fetching villages from MEF API...');
        
        // Clear existing villages
        DB::table('village')->truncate();
        
        $page = 1;
        $totalInserted = 0;
        $totalPages = 1;
        
        do {
            $this->command->info("Fetching page {$page} of {$totalPages}...");
            
            $url = $this->baseUrl . "?page={$page}&page_size={$this->pageSize}";
            
            $response = Http::withOptions([
                'verify' => false,
            ])->timeout(120)->get($url);
            
            if (!$response->successful()) {
                $this->command->error("Failed to fetch page {$page}. Status: " . $response->status());
                $this->command->error("Response: " . $response->body());
                break;
            }
            
            $data = $response->json();
            
            if (empty($data) || !isset($data['items']) || empty($data['items'])) {
                $this->command->info("No more data found on page {$page}.");
                break;
            }
            
            // Update total pages from API response
            $totalPages = $data['total_pages'] ?? 1;
            
            $villages = [];
            foreach ($data['items'] as $item) {
                // Map API fields to database columns
                // API fields: province_code, district_code, commune_code, village_code, village_kh, village_en
                $villages[] = [
                    'id' => (int) $item['village_code'],
                    'commune_id' => (int) $item['commune_code'],
                    'name' => $item['village_kh'] ?? '',
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }
            
            if (!empty($villages)) {
                // Use upsert to handle duplicates
                foreach ($villages as $village) {
                    DB::table('village')->updateOrInsert(
                        ['id' => $village['id']],
                        [
                            'commune_id' => $village['commune_id'],
                            'name' => $village['name'],
                            'updated_at' => $village['updated_at'],
                        ]
                    );
                }
                $totalInserted += count($villages);
                $this->command->info("Processed " . count($villages) . " villages from page {$page}. Total: {$totalInserted}");
            }
            
            $page++;
            
        } while ($page <= $totalPages);
        
        $this->command->info("Village API Seeder completed! Total villages inserted: {$totalInserted}");
    }
}
