<?php

namespace App\Http\Controllers;

use App\Models\Commune;
use App\Models\District;
use App\Models\Education;
use App\Models\EducationYear;
use App\Models\FatherModel;
use App\Models\KudiModel;
use App\Models\MonkType;
use App\Models\MotherModel;
use App\Models\Province;
use App\Models\UserInfoModel;
use App\Models\Village;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\URL;

class UserInfoController extends Controller
{
    public function index(Request $request)
    {
        $sessionKudiId = session('kudi_id');
        $userRole = strtolower(session('user_role') ?? '');
        $isSuperAdmin = in_array($userRole, ['super admin', 'superadmin', 'super_admin']);
        $isAdmin = $userRole === 'admin';

        // Super admin can filter or see all; admin and regular users see their kudi only
        if ($isSuperAdmin) {
            $selectedKudiId = $request->query('kudi_id') ?: null;
        } else {
            // Admin and regular users always see their own kudi
            $selectedKudiId = $sessionKudiId;
        }

        $userinfoQuery = UserInfoModel::with(['province', 'district', 'commune', 'village', 'kudi', 'currentProvince', 'currentDistrict', 'currentCommune', 'currentVillage'])
            ->orderBy('name_kh');

        if ($selectedKudiId) {
            $userinfoQuery->where('kudi_id', $selectedKudiId);
        }

        $userinfo = $userinfoQuery->get();

        // Super admin can see all Kudis for filtering, others only see their assigned Kudi
        $kudisQuery = KudiModel::orderByRaw('CAST(kudi_number AS UNSIGNED)');
        if (!$isSuperAdmin && $sessionKudiId) {
            $kudisQuery->where('id', $sessionKudiId);
        }

        $kudis = $kudisQuery->get();

        // Calculate statistics
        $totalUsers = $userinfo->count();
        $monkCount = $userinfo->filter(function($user) {
            return $user->monkType && $user->monkType->name === 'ភិក្ខុ';
        })->count();
        $noviceCount = $userinfo->filter(function($user) {
            return $user->monkType && $user->monkType->name === 'សាមណេរ';
        })->count();

        $stats = [
            ['label' => 'សរុប', 'value' => $totalUsers, 'icon' => 'users'],
            ['label' => 'ភិក្ខុ', 'value' => $monkCount, 'icon' => 'user'],
            ['label' => 'សាមណេរ', 'value' => $noviceCount, 'icon' => 'user'],
        ];

        // If it's an AJAX request, return only the partial view
        if ($request->ajax()) {
            $view = view('User.partials.user_table', compact('userinfo'))->render();
            return response()->json([
                'success' => true,
                'html' => $view,
                'stats' => $stats // Send stats as array instead of rendered view
            ]);
        }

        return view('User.index', compact(
            'userinfo',
            'kudis',
            'selectedKudiId',
            'stats'
        ));
    }

    /**
     * Get user info details for modal display
     */
    public function getUserDetail($id)
    {
        try {
            \Log::info('getUserDetail called with ID: ' . $id);
            
            // Load all necessary relationships for the modal
            $info = \App\Models\UserInfoModel::with([
                'education',
                'educationYear',
                'monkType',
                'kudi',
                'father',
                'mother',
                'province',
                'district',
                'commune',
                'village',
                'currentProvince',
                'currentDistrict',
                'currentCommune',
                'currentVillage'
            ])->findOrFail($id);
            
            \Log::info('UserInfo found: ' . $info->name_kh);
            \Log::info('Relationships loaded: ' . count($info->getRelations()));
            
            // Check permissions - admin can see all, others only see their kudi
            $userRole = strtolower(session('user_role') ?? '');
            $isAdmin = in_array($userRole, ['admin', 'super admin', 'superadmin', 'super_admin']);
            $sessionKudiId = session('kudi_id');
            
            \Log::info('User role: ' . $userRole . ', Is admin: ' . ($isAdmin ? 'true' : 'false'));
            
            if (!$isAdmin && $sessionKudiId && $info->kudi_id != $sessionKudiId) {
                \Log::info('Permission denied');
                return response()->json([
                    'success' => false,
                    'message' => 'អ្នកមិនមានសិទ្ធិមើលព័ត៌មាននេះទេ។'
                ]);
            }
            
            // Generate HTML for user details
            \Log::info('Rendering view for userinfo details');
            $html = view('User.partials.detail-content', compact('info'))->render();
            \Log::info('View rendered successfully, length: ' . strlen($html));
            
            return response()->json([
                'success' => true,
                'html' => $html
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error in getUserDetail: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'មានបញ្ហាក្នុងការផ្ទុកព័ត៌មាន: ' . $e->getMessage()
            ]);
        }
    }

    public function create(Request $request)
    {
        \Log::info('=== UserInfoController::create() START ===');
        \Log::info('Request method: ' . $request->method());
        \Log::info('Request URL: ' . $request->fullUrl());
        \Log::info('Session data: ' . json_encode(session()->all()));
        \Log::info('User ID in session: ' . session('user_id'));
        \Log::info('User role in session: ' . session('user_role'));
        \Log::info('Kudi ID in session: ' . session('kudi_id'));
        
        try {
            if ($request->isMethod('post')) {
                \Log::info('Processing POST request');
                $this->save($request);
                \Log::info('Save completed, redirecting to userinfo.index');
                return redirect()->route('userinfo.index');
            }

            \Log::info('Preparing form payload for GET request');
            $payload = $this->formPayload($request, new UserInfoModel());
            \Log::info('Form payload keys: ' . json_encode(array_keys($payload)));
            
            \Log::info('Rendering User.form view');
            $response = view('User.form', $payload);
            \Log::info('View rendered successfully');
            \Log::info('=== UserInfoController::create() END ===');
            
            return $response;
            
        } catch (\Exception $e) {
            \Log::error('Error in UserInfoController::create: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            \Log::info('=== UserInfoController::create() ERROR END ===');
            throw $e;
        }
    }

    public function edit(Request $request, $id)
    {
        $userinfo = UserInfoModel::findOrFail($id);

        if ($request->isMethod('post')) {
            $this->save($request, $id);

            $userRole = strtolower(session('user_role') ?? '');
            $isSuperAdmin = in_array($userRole, ['super admin', 'superadmin', 'super_admin']);
            $isAdmin = $userRole === 'admin';

            if ($isSuperAdmin) {
                return redirect()->route('userinfo.index');
            }
            if ($isAdmin) {
                return redirect()->route('userinfo.index');
            }

            return redirect()->route('userinfo.index');
        }

        return view('User.form', $this->formPayload($request, $userinfo));
    }

    private function save(Request $request, $id = null)
    {
        $imageRule = $id ? 'nullable|image|mimes:jpeg,png,jpg|max:3072' : 'required|image|mimes:jpeg,png,jpg|max:3072';

        $validated = $request->validate([
            'name_kh' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'id_number' => 'required|string|max:50',
            'phone_number' => 'nullable|string|max:50',
            'dob' => 'nullable|date',
            'province_id' => 'nullable|exists:province,id',
            'district_id' => 'nullable|exists:district,id',
            'commune_id' => 'nullable|exists:commune,id',
            'village_id' => 'nullable|exists:village,id',
            'temple_name' => 'nullable|string|max:255',
            'current_province_id' => 'nullable|exists:province,id',
            'current_district_id' => 'nullable|exists:district,id',
            'current_commune_id' => 'nullable|exists:commune,id',
            'current_village_id' => 'nullable|exists:village,id',
            'father_id' => 'nullable|exists:father,id',
            'mother_id' => 'nullable|exists:mother,id',
            'father_name' => 'nullable|string|max:255',
            'father_job' => 'nullable|string|max:255',
            'mother_name' => 'nullable|string|max:255',
            'mother_job' => 'nullable|string|max:255',
            'education_id' => 'nullable|exists:education,id',
            'education_year_id' => 'nullable|exists:education_year,id',
            'monk_type_id' => 'nullable|exists:monk_type,id',
            'kudi_id' => 'sometimes|nullable|exists:kudi,id',
            'image' => $imageRule,
        ], [
            'image.max' => 'រូបភាពមានទំហំធំពេក! សូមបញ្ចូលរូបភាពដែលមានទំហំតិចជាង 3MB។',
            'image.mimes' => 'ទម្រង់រូបភាពមិនត្រឹមត្រូវ! សូមប្រើប្រាស់ឯកសារ JPEG, PNG, JPG',
            'image.image' => 'ឯកសារនេះមិនមែនជារូបភាពទេ!',
        ]);

        $userinfo = $id ? UserInfoModel::findOrFail($id) : new UserInfoModel();
        $userinfo->fill(collect($validated)->except('image')->toArray());

        // Auto-set kudi_id = 1 for super admin users
        $sessionUser = \App\Models\UserModel::find(session('user_id'));
        if ($sessionUser && $sessionUser->hasRole('super_admin')) {
            $userinfo->kudi_id = 1;
        }

        if ($request->hasFile('image')) {
            $file = $request->file('image');

            // Check file size before processing
            if ($file->getSize() > 3 * 1024 * 1024) { // 3MB
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['image' => 'រូបភាពមានទំហំធំពេក! សូមបញ្ចូលរូបភាពដែលមានទំហំតិចជាង 3MB។']);
            }

            if ($userinfo->image) {
                Storage::disk('public')->delete($userinfo->image);
            }

            // Use the same compression method as admin users
            $userinfo->image = $this->compressAndSaveImage($file, 'user-info/images');
        }

        $userinfo->save();
    }

    public function publicRegister(Request $request)
    {
        if ($request->isMethod('post')) {
            $userinfo = $this->saveAndReturn($request);
            return redirect()->to(URL::signedRoute('profile.public', ['id' => $userinfo->id]))
                ->with('success', 'ការចុះឈ្មោះបានជោគជ័យ!');
        }

        return view('User.public-form', $this->publicFormPayload($request));
    }

    public function publicProfile($id)
    {
        $userinfo = UserInfoModel::with(['province', 'district', 'commune', 'village', 'education', 'educationYear', 'monkType', 'kudi', 'currentProvince', 'currentDistrict', 'currentCommune', 'currentVillage'])->findOrFail($id);
        return view('User.public-profile', compact('userinfo'));
    }

    public function publicEdit(Request $request, $id)
    {
        $userinfo = UserInfoModel::findOrFail($id);

        if ($request->isMethod('post')) {
            $this->save($request, $id);
            return redirect()->to(URL::signedRoute('profile.public', ['id' => $id]))
                ->with('success', 'កែប្រែព័ត៌មានបានជោគជ័យ!');
        }

        $payload = $this->publicEditPayload($request, $userinfo);
        return view('User.public-edit', $payload);
    }

    public function downloadProfile($id)
    {
        $userinfo = UserInfoModel::with(['province', 'district', 'commune', 'village', 'education', 'educationYear', 'monkType', 'kudi', 'currentProvince', 'currentDistrict', 'currentCommune', 'currentVillage'])->findOrFail($id);

        $pdf = Pdf::loadView('User.profile-pdf', compact('userinfo'))
            ->setPaper('a4', 'portrait')
            ->setOptions([
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
                'isFontSubsettingEnabled' => true,
                'defaultFont' => 'sans-serif',
                'fontHeightRatio' => 1.1,
                'dpi' => 96,
                'enable_php' => false,
                'isPhpEnabled' => false,
                'chroot' => public_path(),
            ]);

        $filename = 'profile-' . $userinfo->id_number . '-' . date('Y-m-d') . '.pdf';

        return $pdf->download($filename);
    }

    private function saveAndReturn(Request $request, $id = null)
    {
        $validated = $request->validate([
            'name_kh' => 'required|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'id_number' => 'required|string|max:50',
            'phone_number' => 'required|string|regex:/^[0-9]{8,15}$/|min:8|max:15',
            'dob' => 'required|date',
            'monk_type_id' => 'required|exists:monk_type,id',
            'kudi_id' => 'required|exists:kudi,id',
            'education_id' => 'required|exists:education,id',
            'education_year_id' => 'nullable|exists:education_year,id',
            'province_id' => 'required|exists:province,id',
            'district_id' => 'required|exists:district,id',
            'commune_id' => 'required|exists:commune,id',
            'village_id' => 'required|exists:village,id',
            'temple_name' => 'required|string|max:255',
            'current_province_id' => 'required|exists:province,id',
            'current_district_id' => 'required|exists:district,id',
            'current_commune_id' => 'required|exists:commune,id',
            'current_village_id' => 'required|exists:village,id',
            'image' => 'required|image|max:2048',
            'mother_id' => 'nullable|exists:mother,id',
            'father_name' => 'nullable|string|max:255',
            'father_job' => 'nullable|string|max:255',
            'mother_name' => 'nullable|string|max:255',
            'mother_job' => 'nullable|string|max:255',
            'education_year_id' => 'nullable|exists:education_year,id',
            'monk_type_id' => 'required|exists:monk_type,id',
            'kudi_id' => 'required|exists:kudi,id',
        ], [
            // Personal Information Messages
            'name_kh.required' => 'សូមបញ្ចូលឈ្មោះជាអក្សរខ្មែរ! ឈ្មោះខ្មែរជាការចាំបាច់សម្រាប់ការចុះឈ្មោះ។',
            'name_kh.string' => 'ឈ្មោះជាអក្សរខ្មែរត្រូវតែជាអក្សរ! សូមបញ្ចូលឈ្មោះត្រឹមត្រូវ។',
            'name_kh.max' => 'ឈ្មោះជាអក្សរខ្មែរមិនអាចលើសពី 255 តួអក្សរទេ! សូមបញ្ចូលឈ្មោះខ្លីជាង។',
            
            'name_en.required' => 'សូមបញ្ចូលឈ្មោះជាអក្សរឡាតាំង! ឈ្មោះឡាតាំងជាការចាំបាច់សម្រាប់ការចុះឈ្មោះ។',
            'name_en.string' => 'ឈ្មោះជាអក្សរឡាតាំងត្រូវតែជាអក្សរ! សូមបញ្ចូលឈ្មោះត្រឹមត្រូវ។',
            'name_en.max' => 'ឈ្មោះជាអក្សរឡាតាំងមិនអាចលើសពី 255 តួអក្សរទេ! សូមបញ្ចូលឈ្មោះខ្លីជាង។',
            
            'id_number.required' => 'សូមបញ្ចូលអត្តលេខ! អត្តលេខជាការចាំបាច់សម្រាប់ការចុះឈ្មោះ។',
            'id_number.string' => 'អត្តលេខត្រូវតែជាអក្សរ/លេខ! សូមបញ្ចូលអត្តលេខត្រឹមត្រូវ។',
            'id_number.max' => 'អត្តលេខមិនអាចលើសពី 50 តួអក្សរទេ! សូមបញ្ចូលអត្តលេខត្រឹមត្រូវ។',
            
            'dob.required' => 'សូមជ្រើសរើសថ្ងៃខែឆ្នាំកំណើត! ថ្ងៃកំណើតជាការចាំបាច់សម្រាប់ការចុះឈ្មោះ។',
            'dob.date' => 'ថ្ងៃខែឆ្នាំកំណើតមិនត្រឹមត្រូវ! សូមជ្រើសរើសថ្ងេខែឆ្នាំកំណើតត្រឹមត្រូវ។',
            
            'monk_type_id.required' => 'សូមជ្រើសរើសប្រភេទភិក្ខុ/សាមណេរ',
            'monk_type_id.exists' => 'ប្រភេទភិក្ខុ/សាមណេរដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសប្រភេទដែលមានក្នុងបញ្ជី។',
            
            // Birth Location Messages
            'province_id.required' => 'សូមជ្រើសរើសខេត្ត/ក្រុងកំណើត! ខេត្ត/ក្រុងកំណើតជាការចាំបាច់សម្រាប់ការចុះឈ្មោះ។',
            'province_id.exists' => 'ខេត្ត/ក្រុងដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសខេត្ត/ក្រុងដែលមានក្នុងបញ្ជី។',
            
            'district_id.required' => 'សូមជ្រើសរើសស្រុក/ខណ្ឌកំណើត! ស្រុក/ខណ្ឌកំណើតជាការចាំបាច់សម្រាប់ការចុះឈ្មោះ។',
            'district_id.exists' => 'ស្រុក/ខណ្ឌដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសស្រុក/ខណ្ឌដែលមានក្នុងបញ្ជី។',
            
            'commune_id.required' => 'សូមជ្រើសរើសឃុំ/សង្កាត់កំណើត! ឃុំ/សង្កាត់កំណើតជាការចាំបាច់សម្រាប់ការចុះឈ្មោះ។',
            'commune_id.exists' => 'ឃុំ/សង្កាត់ដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសឃុំ/សង្កាត់ដែលមានក្នុងបញ្ជី។',
            
            'village_id.required' => 'សូមជ្រើសរើសភូមិកំណើត! ភូមិកំណើតជាការចាំបាច់សម្រាប់ការចុះឈ្មោះ។',
            'village_id.exists' => 'ភូមិដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសភូមិដែលមានក្នុងបញ្ជី។',
            
            // Education Messages
            'education_id.required' => 'សូមជ្រើសរើសកម្រិតវប្បធម៌! កម្រិតវប្បធម៌ជាការចាំបាច់សម្រាប់ការចុះឈ្មោះ។',
            'education_id.exists' => 'កម្រិតវប្បធម៌ដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសកម្រិតវប្បធម៌ដែលមានក្នុងបញ្ជី។',
            
            'education_year_id.exists' => 'ឆ្នាំសិក្សាដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសឆ្នាំសិក្សាដែលមានក្នុងបញ្ជី។',
            
            // Kudi Messages
            'kudi_id.required' => 'សូមជ្រើសរើសកុដិ! កុដិជាការចាំបាច់សម្រាប់ការចុះឈ្មោះ។',
            'kudi_id.exists' => 'កុដិដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសកុដិដែលមានក្នុងបញ្ជី។',
            
            // Image Messages
            'image.required' => 'សូមបញ្ចូលរូបភាព',
            'image.image' => 'ឯកសារដែលបានជ្រើសរើសត្រូវតែជារូបភាព! សូមជ្រើសរើសឯកសាររូបភាព (JPG, PNG, GIF)។',
            'image.max' => 'ទំហំរូបភាពមិនអាចលើសពី 2MB ទេ! សូមជ្រើសរើសរូបភាពដែលមានទំហំតិចជាង 2MB។',
            
            // Phone Number Messages
            'phone_number.required' => 'សូមបញ្ចូលលេខទូរស័ព្ទ! លេខទូរស័ព្ទជាការចាំបាច់សម្រាប់ការចុះឈ្មោះ។',
            'phone_number.string' => 'លេខទូរស័ព្ទត្រូវតែជាអក្សរ/លេខ! សូមបញ្ចូលលេខទូរស័ព្ទត្រឹមត្រូវ។',
            'phone_number.regex' => 'លេខទូរស័ព្ទត្រូវតែជាលេខគត់ និងមានចាន់ពី 8 ដល់ 15 ខ្ទង់! សូមបញ្ចូលលេខទូរស័ព្ទត្រឹមត្រូវ។',
            'phone_number.min' => 'លេខទូរស័ព្ទត្រូវមានចាន់ពី 8 ខ្ទង់! សូមបញ្ចូលលេខទូរស័ព្ទត្រឹមត្រូវ។',
            'phone_number.max' => 'លេខទូរស័ព្ទមិនអាចលើសពី 15 ខ្ទង់! សូមបញ្ចូលលេខទូរស័ព្ទត្រឹមត្រូវ។',
            
            'temple_name.string' => 'ឈ្មោះវត្តត្រូវតែជាអក្សរ! សូមបញ្ចូលឈ្មោះវត្តត្រឹមត្រូវ។',
            'temple_name.max' => 'ឈ្មោះវត្តមិនអាចលើសពី 255 តួអក្សរទេ! សូមបញ្ចូលឈ្មោះវត្តខ្លីជាង។',
            
            'father_name.string' => 'ឈ្មោះឪពុកត្រូវតែជាអក្សរ! សូមបញ្ចូលឈ្មោះឪពុកត្រឹមត្រូវ។',
            'father_name.max' => 'ឈ្មោះឪពុកមិនអាចលើសពី 255 តួអក្សរទេ! សូមបញ្ចូលឈ្មោះឪពុកខ្លីជាង។',
            
            'father_job.string' => 'មុខរបរឪពុកត្រូវតែជាអក្សរ! សូមបញ្ចូលមុខរបរឪពុកត្រឹមត្រូវ។',
            'father_job.max' => 'មុខរបរឪពុកមិនអាចលើសពី 255 តួអក្សរទេ! សូមបញ្ចូលមុខរបរឪពុកខ្លីជាង។',
            
            'mother_name.string' => 'ឈ្មោះម្ដាយត្រូវតែជាអក្សរ! សូមបញ្ចូលឈ្មោះម្ដាយត្រឹមត្រូវ។',
            'mother_name.max' => 'ឈ្មោះម្ដាយមិនអាចលើសពី 255 តួអក្សរទេ! សូមបញ្ចូលឈ្មោះម្ដាយខ្លីជាង។',
            
            'mother_job.string' => 'មុខរបរម្ដាយត្រូវតែជាអក្សរ! សូមបញ្ចូលមុខរបរម្ដាយត្រឹមត្រូវ។',
            'mother_job.max' => 'មុខរបរម្ដាយមិនអាចលើសពី 255 តួអក្សរទេ! សូមបញ្ចូលមុខរបរម្ដាយខ្លីជាង។',
            
            // Current Address Messages
            'current_province_id.exists' => 'ខេត្ត/ក្រុងបច្ចុប្បន្នដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសខេត្ត/ក្រុងដែលមានក្នុងបញ្ជី។',
            'current_district_id.exists' => 'ស្រុក/ខណ្ឌបច្ចុប្បន្នដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសស្រុក/ខណ្ឌដែលមានក្នុងបញ្ជី។',
            'current_commune_id.exists' => 'ឃុំ/សង្កាត់បច្ចុប្បន្នដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសឃុំ/សង្កាត់ដែលមានក្នុងបញ្ជី។',
            'current_village_id.exists' => 'ភូមិបច្ចុប្បន្នដែលជ្រើសរើសមិនមានក្នុងប្រព័ន្ធ! សូមជ្រើសរើសភូមិដែលមានក្នុងបញ្ជី។',
        ]);

        $userinfo = $id ? UserInfoModel::findOrFail($id) : new UserInfoModel();
        $userinfo->fill(collect($validated)->except('image')->toArray());

        if ($request->hasFile('image')) {
            if ($userinfo->image) {
                Storage::disk('public')->delete($userinfo->image);
            }
            $userinfo->image = $request->file('image')->store('user-info/images', 'public');
        }

        $userinfo->save();
        return $userinfo;
    }

    private function publicEditPayload(Request $request, UserInfoModel $userinfo): array
    {
        $provinceId = old('province_id', $userinfo->province_id);
        $districtId = old('district_id', $userinfo->district_id);
        $communeId = old('commune_id', $userinfo->commune_id);
        $educationId = old('education_id', $userinfo->education_id);

        $currentProvinceId = old('current_province_id', $userinfo->current_province_id);
        $currentDistrictId = old('current_district_id', $userinfo->current_district_id);
        $currentCommuneId = old('current_commune_id', $userinfo->current_commune_id);

        return [
            'userinfo' => $userinfo,
            'provinces' => Province::orderBy('name')->get(),
            'districts' => $provinceId
                ? District::where('province_id', $provinceId)->orderBy('name')->get()
                : collect(),
            'communes' => $districtId
                ? Commune::where('district_id', $districtId)->orderBy('name')->get()
                : collect(),
            'villages' => $communeId
                ? Village::where('commune_id', $communeId)->orderBy('name')->get()
                : collect(),
            'currentDistricts' => $currentProvinceId
                ? District::where('province_id', $currentProvinceId)->orderBy('name')->get()
                : collect(),
            'currentCommunes' => $currentDistrictId
                ? Commune::where('district_id', $currentDistrictId)->orderBy('name')->get()
                : collect(),
            'currentVillages' => $currentCommuneId
                ? Village::where('commune_id', $currentCommuneId)->orderBy('name')->get()
                : collect(),
            'educations' => Education::orderBy('name')->get(),
            'years' => $educationId
                ? EducationYear::where('education_id', $educationId)->orderBy('name')->get()
                : collect(),
            'monkTypes' => MonkType::orderBy('name')->get(),
            'kudis' => KudiModel::orderByRaw('CAST(kudi_number AS UNSIGNED)')->get(),
        ];
    }

    private function publicFormPayload(Request $request): array
    {
        $provinceId = old('province_id');
        $districtId = old('district_id');
        $communeId = old('commune_id');
        $educationId = old('education_id');

        // Current address values
        $currentProvinceId = old('current_province_id');
        $currentDistrictId = old('current_district_id');
        $currentCommuneId = old('current_commune_id');

        return [
            'userinfo' => new UserInfoModel(),
            'provinces' => Province::orderBy('name')->get(),
            'districts' => $provinceId
                ? District::where('province_id', $provinceId)->orderBy('name')->get()
                : collect(),
            'communes' => $districtId
                ? Commune::where('district_id', $districtId)->orderBy('name')->get()
                : collect(),
            'villages' => $communeId
                ? Village::where('commune_id', $communeId)->orderBy('name')->get()
                : collect(),
            'currentDistricts' => $currentProvinceId
                ? District::where('province_id', $currentProvinceId)->orderBy('name')->get()
                : collect(),
            'currentCommunes' => $currentDistrictId
                ? Commune::where('district_id', $currentDistrictId)->orderBy('name')->get()
                : collect(),
            'currentVillages' => $currentCommuneId
                ? Village::where('commune_id', $currentCommuneId)->orderBy('name')->get()
                : collect(),
            'educations' => Education::orderBy('name')->get(),
            'years' => $educationId
                ? EducationYear::where('education_id', $educationId)->orderBy('name')->get()
                : collect(),
            'monkTypes' => MonkType::orderBy('name')->get(),
            'kudis' => KudiModel::orderByRaw('CAST(kudi_number AS UNSIGNED)')->get(),
        ];
    }

    public function delete($id)
    {
        $userinfo = UserInfoModel::findOrFail($id);

        // Delete image file before deleting the database record
        if ($userinfo->image) {
            if (Storage::disk('public')->exists($userinfo->image)) {
                Storage::disk('public')->delete($userinfo->image);
            }

            $dir = dirname($userinfo->image);
            if ($dir !== '.' && $dir !== '') {
                $files = Storage::disk('public')->files($dir);
                $subdirs = Storage::disk('public')->allDirectories($dir);
                if (empty($files) && empty($subdirs)) {
                    Storage::disk('public')->deleteDirectory($dir);
                }
            }
        }

        // Now delete the database record
        $userinfo->delete();

        return redirect()->route('userinfo.index');
    }

    private function formPayload(Request $request, UserInfoModel $userinfo): array
    {
        $provinceId = $this->selectedValue($request, 'province_id', $userinfo);
        $districtId = $this->selectedValue($request, 'district_id', $userinfo);
        $communeId = $this->selectedValue($request, 'commune_id', $userinfo);
        $educationId = $this->selectedValue($request, 'education_id', $userinfo);

        // Current address values
        $currentProvinceId = $this->selectedValue($request, 'current_province_id', $userinfo);
        $currentDistrictId = $this->selectedValue($request, 'current_district_id', $userinfo);
        $currentCommuneId = $this->selectedValue($request, 'current_commune_id', $userinfo);

        $sessionKudiId = session('kudi_id');
        $userRole = strtolower(session('user_role') ?? '');
        $isAdmin = in_array($userRole, ['admin', 'super admin', 'superadmin', 'super_admin']);

        // Admin can see all Kudis, regular users only see their assigned Kudi
        $kudisQuery = KudiModel::orderByRaw('CAST(kudi_number AS UNSIGNED)');
        if (!$isAdmin && $sessionKudiId) {
            $kudisQuery->where('id', $sessionKudiId);
        }

        return [
            'userinfo' => $userinfo,
            'provinces' => Province::orderBy('name')->get(),
            'districts' => $provinceId
                ? District::where('province_id', $provinceId)->orderBy('name')->get()
                : collect(),
            'communes' => $districtId
                ? Commune::where('district_id', $districtId)->orderBy('name')->get()
                : collect(),
            'villages' => $communeId
                ? Village::where('commune_id', $communeId)->orderBy('name')->get()
                : collect(),
            'educations' => Education::orderBy('name')->get(),
            'years' => $educationId
                ? EducationYear::where('education_id', $educationId)->orderBy('name')->get()
                : collect(),
            'monkTypes' => MonkType::orderBy('name')->get(),
            'kudis' => $kudisQuery->get(),
            'fathers' => FatherModel::orderBy('name')->get(),
            'mothers' => MotherModel::orderBy('name')->get(),
            'currentDistricts' => $currentProvinceId
                ? District::where('province_id', $currentProvinceId)->orderBy('name')->get()
                : collect(),
            'currentCommunes' => $currentDistrictId
                ? Commune::where('district_id', $currentDistrictId)->orderBy('name')->get()
                : collect(),
            'currentVillages' => $currentCommuneId
                ? Village::where('commune_id', $currentCommuneId)->orderBy('name')->get()
                : collect(),
            'isAdmin' => $isAdmin,
            'sessionKudiId' => $sessionKudiId,
        ];
    }

    private function selectedValue(Request $request, string $field, UserInfoModel $userinfo): mixed
    {
        $value = old($field);

        if ($value === null) {
            $value = $request->input($field, $userinfo->{$field} ?? null);
        }

        return $value === '' ? null : $value;
    }

    private function compressAndSaveImage($file, $directory = 'user-info/images')
    {
        $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
        $path = $directory . '/' . $filename;

        // Check if GD library is available
        if (!extension_loaded('gd') || !function_exists('gd_info')) {
            // GD library not available, save original file
            return $file->store($directory, 'public');
        }

        try {
            // Get image info
            $imageInfo = getimagesize($file);
            if (!$imageInfo) {
                return $file->store($directory, 'public');
            }

            $mime = $imageInfo['mime'];

            // Create image resource based on mime type
            switch ($mime) {
                case 'image/jpeg':
                    if (!function_exists('imagecreatefromjpeg')) {
                        return $file->store($directory, 'public');
                    }
                    $image = imagecreatefromjpeg($file);
                    break;
                case 'image/png':
                    if (!function_exists('imagecreatefrompng')) {
                        return $file->store($directory, 'public');
                    }
                    $image = imagecreatefrompng($file);
                    break;
                case 'image/jpg':
                    if (!function_exists('imagecreatefromjpeg')) {
                        return $file->store($directory, 'public');
                    }
                    $image = imagecreatefromjpeg($file);
                    break;
                default:
                    // If unsupported format, just save as is
                    return $file->store($directory, 'public');
            }

            // Get original dimensions
            $width = imagesx($image);
            $height = imagesy($image);

            // Calculate new dimensions (max width 800px, max height 800px)
            $maxWidth = 800;
            $maxHeight = 800;

            if ($width > $maxWidth || $height > $maxHeight) {
                $ratio = min($maxWidth / $width, $maxHeight / $height);
                $newWidth = (int)($width * $ratio);
                $newHeight = (int)($height * $ratio);

                // Check if required functions exist
                if (!function_exists('imagecreatetruecolor') || !function_exists('imagecopyresampled')) {
                    imagedestroy($image);
                    return $file->store($directory, 'public');
                }

                // Create new image
                $newImage = imagecreatetruecolor($newWidth, $newHeight);

                // Preserve transparency for PNG
                if ($mime == 'image/png') {
                    if (function_exists('imagealphablending') && function_exists('imagesavealpha') && function_exists('imagecolorallocatealpha')) {
                        imagealphablending($newImage, false);
                        imagesavealpha($newImage, true);
                        $transparent = imagecolorallocatealpha($newImage, 255, 255, 255, 127);
                        if (function_exists('imagefilledrectangle')) {
                            imagefilledrectangle($newImage, 0, 0, $newWidth, $newHeight, $transparent);
                        }
                    }
                }

                // Resize image
                imagecopyresampled($newImage, $image, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);

                // Save compressed image
                $tempPath = storage_path('app/public/temp_' . $filename);

                switch ($mime) {
                    case 'image/jpeg':
                        if (function_exists('imagejpeg')) {
                            imagejpeg($newImage, $tempPath, 85);
                        } else {
                            imagedestroy($newImage);
                            imagedestroy($image);
                            return $file->store($directory, 'public');
                        }
                        break;
                    case 'image/png':
                        if (function_exists('imagepng')) {
                            imagepng($newImage, $tempPath, 8);
                        } else {
                            imagedestroy($newImage);
                            imagedestroy($image);
                            return $file->store($directory, 'public');
                        }
                        break;
                    case 'image/jpg':
                        if (function_exists('imagejpeg')) {
                            imagejpeg($newImage, $tempPath, 85);
                        } else {
                            imagedestroy($newImage);
                            imagedestroy($image);
                            return $file->store($directory, 'public');
                        }
                        break;
                }

                // Move to final location
                Storage::disk('public')->put($path, file_get_contents($tempPath));

                // Clean up
                unlink($tempPath);
                imagedestroy($newImage);
            } else {
                // Image is already small enough, just save with compression
                $tempPath = storage_path('app/public/temp_' . $filename);

                switch ($mime) {
                    case 'image/jpeg':
                        if (function_exists('imagejpeg')) {
                            imagejpeg($image, $tempPath, 85);
                        } else {
                            imagedestroy($image);
                            return $file->store($directory, 'public');
                        }
                        break;
                    case 'image/png':
                        if (function_exists('imagepng')) {
                            imagepng($image, $tempPath, 8);
                        } else {
                            imagedestroy($image);
                            return $file->store($directory, 'public');
                        }
                        break;
                    case 'image/jpg':
                        if (function_exists('imagejpeg')) {
                            imagejpeg($image, $tempPath, 85);
                        } else {
                            imagedestroy($image);
                            return $file->store($directory, 'public');
                        }
                        break;
                }

                Storage::disk('public')->put($path, file_get_contents($tempPath));
                unlink($tempPath);
            }

            imagedestroy($image);
            return $path;

        } catch (\Exception $e) {
            // If compression fails, save original
            return $file->store($directory, 'public');
        }
    }
}
