<?php

namespace App\Http\Controllers;

use App\Models\KudiModel;
use App\Models\UserModel;
use App\Models\UserInfoModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Routing\Controller;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $sessionKudiId = session('kudi_id');
        $userRole = strtolower(session('user_role') ?? '');
        $isAdmin = in_array($userRole, ['admin', 'super admin', 'superadmin', 'super_admin']);
        
        $selectedKudiId = $request->query('kudi_id') ?: ($isAdmin ? null : $sessionKudiId);

        $usersQuery = UserModel::with(['kudi'])->orderBy('name');

        if ($selectedKudiId) {
            $usersQuery->where('kudi_id', $selectedKudiId);
        } elseif (!$isAdmin && $sessionKudiId) {
            $usersQuery->where('kudi_id', $sessionKudiId);
        }

        $users = $usersQuery->get();
        
        // Admin can see all Kudis, regular users only see their assigned Kudi
        $kudisQuery = KudiModel::orderByRaw('CAST(kudi_number AS UNSIGNED)');
        if (!$isAdmin && $sessionKudiId) {
            $kudisQuery->where('id', $sessionKudiId);
        }
        $kudis = $kudisQuery->get();

        // If it's an AJAX request, return JSON response
        if ($request->ajax()) {
            $view = view('admin.user.partials.user_table', compact('users'))->render();
            return response()->json([
                'success' => true,
                'html' => $view,
                'users_count' => $users->count()
            ]);
        }

        return view('admin.user.index', compact('users', 'kudis', 'selectedKudiId', 'isAdmin'));
    }
    public function create(Request $request)
    {
        if ($request->isMethod('post')) {
            $this->save($request);
            return redirect()->route('user.index');
        }
        
        $sessionKudiId = session('kudi_id');
        $userRole = strtolower(session('user_role') ?? '');
        $isAdmin = in_array($userRole, ['admin', 'super admin', 'superadmin', 'super_admin']);
        
        // Admin can see all Kudis, regular users only see their assigned Kudi
        $kudisQuery = KudiModel::orderByRaw('CAST(kudi_number AS UNSIGNED)');
        if (!$isAdmin && $sessionKudiId) {
            $kudisQuery->where('id', $sessionKudiId);
        }
        $kudi = $kudisQuery->get();

        return view('admin.user.form', compact('kudi', 'isAdmin', 'sessionKudiId'));
    }
    public function edit(Request $request, $id)
    {
        $user = UserModel::find($id);
        if ($request->isMethod('post')) {
            $this->save($request, $id);
            return redirect()->route('user.index');
        }
        
        $sessionKudiId = session('kudi_id');
        $userRole = strtolower(session('user_role') ?? '');
        $isAdmin = in_array($userRole, ['admin', 'super admin', 'superadmin', 'super_admin']);
        
        // Admin can see all Kudis, regular users only see their assigned Kudi
        $kudisQuery = KudiModel::orderByRaw('CAST(kudi_number AS UNSIGNED)');
        if (!$isAdmin && $sessionKudiId) {
            $kudisQuery->where('id', $sessionKudiId);
        }
        $kudi = $kudisQuery->get();
        
        return view('admin.user.form', compact('user', 'kudi', 'isAdmin', 'sessionKudiId'));
    }

    /**
     * Get user details for modal display
     */
    public function getUserDetail($id)
    {
        try {
            \Log::info('getUserDetail called with ID: ' . $id);
            
            $user = UserModel::findOrFail($id);
            \Log::info('User found: ' . $user->name);
            
            // Check permissions - admin can see all, others only see their kudi
            $userRole = strtolower(session('user_role') ?? '');
            $isAdmin = in_array($userRole, ['admin', 'super admin', 'superadmin', 'super_admin']);
            $sessionKudiId = session('kudi_id');
            
            \Log::info('User role: ' . $userRole . ', Is admin: ' . ($isAdmin ? 'true' : 'false'));
            
            if (!$isAdmin && $sessionKudiId && $user->kudi_id != $sessionKudiId) {
                \Log::info('Permission denied');
                return response()->json([
                    'success' => false,
                    'message' => 'អ្នកមិនមានសិទ្ធិមើលព័ត៌មាននេះទេ។'
                ]);
            }
            
            // Generate HTML for user details
            \Log::info('Rendering view for user details');
            $html = view('admin.user.detail-modal', compact('user'))->render();
            \Log::info('View rendered successfully, length: ' . strlen($html));
            
            return response()->json([
                'success' => true,
                'html' => $html
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error in getUserDetail: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'មានបញ្ហាក្នុងការផ្ទុកព័ត៌មាន: ' . $e->getMessage()
            ]);
        }
    }


   private function save(Request $request, $id = null)
{
    // 1. Validate the data
    $validationRules = [
        'name' => 'required',
        'kudi_id' => 'sometimes|required|exists:kudi,id',
        'role' => 'nullable',
        'image' => 'nullable|image|mimes:jpeg,png,jpg|max:3072',
    ];
    
    // Make kudi_id not required for super_admin
    if ($request->input('role') === 'super_admin') {
        $validationRules['kudi_id'] = 'nullable|exists:kudi,id';
    }
    
    $validated = $request->validate($validationRules, [
        'image.max' => 'រូបភាពមានទំហំធំពេក! សូមបញ្ចូលរូបភាពដែលមានទំហំតិចជាង 3MB។',
        'image.mimes' => 'ទម្រង់រូបភាពមិនត្រឹមត្រូវ! សូមប្រើប្រាស់ឯកសារ JPEG, PNG, JPG, GIF ឬ WebP។',
        'image.image' => 'ឯកសារនេះមិនមែនជារូបភាពទេ!',
    ]);

    // 2. Prepare the data array
    $data = $request->only(['name', 'role', 'kudi_id']);

    // 3. Handle Instance (Find for edit, or New for create)
    $user = $id ? UserModel::find($id) : new UserModel();

    // 4. Handle Password (if provided)
    if ($request->filled('password')) {
        $data['password'] = Hash::make($request->password);
    }

    // 5. Handle Image Upload with Compression
    if ($request->hasFile('image')) {
        $file = $request->file('image');
        
        // Check file size before processing
        if ($file->getSize() > 3 * 1024 * 1024) { // 3MB
            return redirect()->back()
                ->withInput()
                ->withErrors(['image' => 'រូបភាពមានទំហំធំពេក! សូមបញ្ចូលរូបភាពដែលមានទំហំតិចជាង 3MB។']);
        }

        // Delete old image if exists
        if ($id && $user->image) {
            Storage::disk('public')->delete($user->image);
        }

        // Compress and save image
        $data['image'] = $this->compressAndSaveImage($file);
    }

    // 6. THE CRITICAL PART: Insert or Update the Database
    if ($id) {
        $user->update($data); // Updates existing user
    } else {
        UserModel::create($data); // Inserts new user
    }
}

private function compressAndSaveImage($file)
{
    $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
    $path = 'users/images/' . $filename;
    
    // Check if GD library is available
    if (!extension_loaded('gd') || !function_exists('gd_info')) {
        // GD library not available, save original file
        return $file->store('users/images', 'public');
    }
    
    try {
        // Get image info
        $imageInfo = getimagesize($file);
        if (!$imageInfo) {
            return $file->store('users/images', 'public');
        }
        
        $mime = $imageInfo['mime'];
        
        // Create image resource based on mime type
        switch ($mime) {
            case 'image/jpeg':
                if (!function_exists('imagecreatefromjpeg')) {
                    return $file->store('users/images', 'public');
                }
                $image = imagecreatefromjpeg($file);
                break;
            case 'image/png':
                if (!function_exists('imagecreatefrompng')) {
                    return $file->store('users/images', 'public');
                }
                $image = imagecreatefrompng($file);
                break;
            case 'image/gif':
                if (!function_exists('imagecreatefromgif')) {
                    return $file->store('users/images', 'public');
                }
                $image = imagecreatefromgif($file);
                break;
            case 'image/webp':
                if (!function_exists('imagecreatefromwebp')) {
                    return $file->store('users/images', 'public');
                }
                $image = imagecreatefromwebp($file);
                break;
            default:
                // If unsupported format, just save as is
                return $file->store('users/images', 'public');
        }
        
        // Get original dimensions
        $width = imagesx($image);
        $height = imagesy($image);
        
        // Calculate new dimensions (max width 800px, max height 800px)
        $maxWidth = 800;
        $maxHeight = 800;
        
        if ($width > $maxWidth || $height > $maxHeight) {
            $ratio = min($maxWidth / $width, $maxHeight / $height);
            $newWidth = (int)($width * $ratio);
            $newHeight = (int)($height * $ratio);
            
            // Check if required functions exist
            if (!function_exists('imagecreatetruecolor') || !function_exists('imagecopyresampled')) {
                imagedestroy($image);
                return $file->store('users/images', 'public');
            }
            
            // Create new image
            $newImage = imagecreatetruecolor($newWidth, $newHeight);
            
            // Preserve transparency for PNG and GIF
            if ($mime == 'image/png' || $mime == 'image/gif') {
                if (function_exists('imagealphablending') && function_exists('imagesavealpha') && function_exists('imagecolorallocatealpha')) {
                    imagealphablending($newImage, false);
                    imagesavealpha($newImage, true);
                    $transparent = imagecolorallocatealpha($newImage, 255, 255, 255, 127);
                    if (function_exists('imagefilledrectangle')) {
                        imagefilledrectangle($newImage, 0, 0, $newWidth, $newHeight, $transparent);
                    }
                }
            }
            
            // Resize image
            imagecopyresampled($newImage, $image, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
            
            // Save compressed image
            $tempPath = storage_path('app/public/temp_' . $filename);
            
            switch ($mime) {
                case 'image/jpeg':
                    if (function_exists('imagejpeg')) {
                        imagejpeg($newImage, $tempPath, 85);
                    } else {
                        imagedestroy($newImage);
                        imagedestroy($image);
                        return $file->store('users/images', 'public');
                    }
                    break;
                case 'image/png':
                    if (function_exists('imagepng')) {
                        imagepng($newImage, $tempPath, 8);
                    } else {
                        imagedestroy($newImage);
                        imagedestroy($image);
                        return $file->store('users/images', 'public');
                    }
                    break;
                case 'image/gif':
                    if (function_exists('imagegif')) {
                        imagegif($newImage, $tempPath);
                    } else {
                        imagedestroy($newImage);
                        imagedestroy($image);
                        return $file->store('users/images', 'public');
                    }
                    break;
                case 'image/webp':
                    if (function_exists('imagewebp')) {
                        imagewebp($newImage, $tempPath, 85);
                    } else {
                        imagedestroy($newImage);
                        imagedestroy($image);
                        return $file->store('users/images', 'public');
                    }
                    break;
            }
            
            // Move to final location
            Storage::disk('public')->put($path, file_get_contents($tempPath));
            
            // Clean up
            unlink($tempPath);
            imagedestroy($newImage);
        } else {
            // Image is already small enough, just save with compression
            $tempPath = storage_path('app/public/temp_' . $filename);
            
            switch ($mime) {
                case 'image/jpeg':
                    if (function_exists('imagejpeg')) {
                        imagejpeg($image, $tempPath, 85);
                    } else {
                        imagedestroy($image);
                        return $file->store('users/images', 'public');
                    }
                    break;
                case 'image/png':
                    if (function_exists('imagepng')) {
                        imagepng($image, $tempPath, 8);
                    } else {
                        imagedestroy($image);
                        return $file->store('users/images', 'public');
                    }
                    break;
                case 'image/gif':
                    if (function_exists('imagegif')) {
                        imagegif($image, $tempPath);
                    } else {
                        imagedestroy($image);
                        return $file->store('users/images', 'public');
                    }
                    break;
                case 'image/webp':
                    if (function_exists('imagewebp')) {
                        imagewebp($image, $tempPath, 85);
                    } else {
                        imagedestroy($image);
                        return $file->store('users/images', 'public');
                    }
                    break;
            }
            
            Storage::disk('public')->put($path, file_get_contents($tempPath));
            unlink($tempPath);
        }
        
        imagedestroy($image);
        return $path;
        
    } catch (\Exception $e) {
        // If compression fails, save original
        return $file->store('users/images', 'public');
    }
}


    public function delete($id)
    {
        $user = UserModel::find($id);


        if ($user->image) {
            if (Storage::disk('public')->exists($user->image)) {
                Storage::disk('public')->delete($user->image);
            }

            $dir = dirname($user->image);
            if ($dir !== '.' && $dir !== '') {
                $files = Storage::disk('public')->files($dir);
                $subdirs = Storage::disk('public')->allDirectories($dir);
                if (empty($files) && empty($subdirs)) {
                    Storage::disk('public')->deleteDirectory($dir);
                }
            }
        }

        $user->delete();
        return redirect()->route('user.index');
    }
}
